"use strict";
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Any modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.relativeToRepoRoot = exports.getRepoRoot = exports.getMatchingRoot = exports.UPSTREAM_BRANCH = exports.REPO_ROOT_8_3 = exports.REPO_ROOT = void 0;
const tslib_1 = require("tslib");
/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
const path_1 = require("path");
const load_json_file_1 = tslib_1.__importDefault(require("load-json-file"));
const path_2 = require("./path");
const readOpenSearchDashboardsPkgJson = (dir) => {
    try {
        const path = (0, path_1.resolve)(dir, 'package.json');
        const json = load_json_file_1.default.sync(path);
        if (json?.name === 'opensearch-dashboards') {
            return json;
        }
    }
    catch (error) {
        if (error?.code === 'ENOENT') {
            return;
        }
        throw error;
    }
};
const findOpenSearchDashboardsPackageJson = () => {
    // search for the opensearch-dashboards directory, since this file is moved around it might
    // not be where we think but should always be a relatively close parent
    // of this directory
    const startDir = (0, path_2.realPathSync)(__dirname);
    const { root: rootDir } = (0, path_1.parse)(startDir);
    let cursor = startDir;
    while (true) {
        const opensearchDashboardsPkgJson = readOpenSearchDashboardsPkgJson(cursor);
        if (opensearchDashboardsPkgJson) {
            return {
                opensearchDashboardsDir: cursor,
                opensearchDashboardsPkgJson: opensearchDashboardsPkgJson,
            };
        }
        const parent = (0, path_1.dirname)(cursor);
        if (parent === rootDir) {
            throw new Error(`unable to find opensearch-dashboards directory from ${startDir}`);
        }
        cursor = parent;
    }
};
const { opensearchDashboardsDir, opensearchDashboardsPkgJson, } = findOpenSearchDashboardsPackageJson();
exports.REPO_ROOT = (0, path_2.resolveToFullPathSync)(opensearchDashboardsDir);
exports.REPO_ROOT_8_3 = (0, path_2.resolveToShortPathSync)(opensearchDashboardsDir);
exports.UPSTREAM_BRANCH = opensearchDashboardsPkgJson.branch;
const getMatchingRoot = (path, rootPaths) => {
    const rootPathsArray = Array.isArray(rootPaths) ? rootPaths : [rootPaths];
    // We can only find the appropriate root if an absolute path was given
    if (path && (0, path_1.isAbsolute)(path)) {
        // Return the matching root if one is found or return `undefined`
        return rootPathsArray.find((root) => path.startsWith(root));
    }
    return undefined;
};
exports.getMatchingRoot = getMatchingRoot;
const getRepoRoot = (path) => (0, exports.getMatchingRoot)(path, [exports.REPO_ROOT, exports.REPO_ROOT_8_3]);
exports.getRepoRoot = getRepoRoot;
const relativeToRepoRoot = (path) => {
    const repoRoot = (0, exports.getRepoRoot)(path);
    return repoRoot ? (0, path_1.relative)(repoRoot, path) : null;
};
exports.relativeToRepoRoot = relativeToRepoRoot;
