"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.extractDatasetTypeFromUri = exports.extractDataSourceInfoFromUri = exports.constructDataSourceUri = exports.concatDataSourceWithDataView = void 0;
exports.findByTitle = findByTitle;
exports.validateDataViewDataSourceReference = exports.getDatasetTypeFromReference = exports.getDataViewTitle = exports.getDataSourceReference = void 0;
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * Returns an object matching a given title
 *
 * @param client {DataViewSavedObjectsClientCommon}
 * @param title {string}
 * @param dataSourceId {string}{optional}
 * @returns {Promise<SavedObject|undefined>}
 */
async function findByTitle(client, title, dataSourceId) {
  if (title) {
    const savedObjects = (await client.find({
      type: 'index-pattern',
      perPage: 10,
      search: `"${title}"`,
      searchFields: ['title'],
      fields: ['title']
    })).filter(obj => {
      return obj && obj.attributes && validateDataViewDataSourceReference(obj, dataSourceId);
    });
    return savedObjects.find(obj => obj.attributes.title.toLowerCase() === title.toLowerCase());
  }
}
const validateDataViewDataSourceReference = (dataView, dataSourceId) => {
  const references = dataView.references;
  if (dataSourceId) {
    return references.some(ref => ref.id === dataSourceId && ref.type === 'data-source');
  } else {
    return references.length === 0;
  }
};
exports.validateDataViewDataSourceReference = validateDataViewDataSourceReference;
const getDataViewTitle = async (dataViewTitle, references, getDataSource) => {
  let dataSourceTitle;
  const dataSourceReference = getDataSourceReference(references);
  if (dataSourceReference) {
    const dataSourceId = dataSourceReference.id;
    try {
      const {
        attributes: {
          title
        },
        error
      } = await getDataSource(dataSourceId);
      dataSourceTitle = error ? dataSourceId : title;
    } catch (e) {
      dataSourceTitle = dataSourceId;
    }
    return concatDataSourceWithDataView(dataSourceTitle, dataViewTitle);
  } else {
    return dataViewTitle;
  }
};
exports.getDataViewTitle = getDataViewTitle;
const concatDataSourceWithDataView = (dataSourceTitle, dataViewTitle) => {
  const DATA_SOURCE_DATA_VIEW_DELIMITER = '::';
  return dataSourceTitle.concat(DATA_SOURCE_DATA_VIEW_DELIMITER).concat(dataViewTitle);
};
exports.concatDataSourceWithDataView = concatDataSourceWithDataView;
const getDataSourceReference = references => {
  return references.find(ref => ref.type === 'data-source');
};

/**
 * Extracts dataset type from a URI pattern
 * @param uri - URI in format "type://name" (e.g., "s3://my-bucket", "index-pattern://logs-*")
 * @returns The dataset type in uppercase or undefined if not found
 */
exports.getDataSourceReference = getDataSourceReference;
const extractDatasetTypeFromUri = uri => {
  if (!uri || !uri.includes('://')) {
    return undefined;
  }
  const [type] = uri.split('://');
  return type === null || type === void 0 ? void 0 : type.toUpperCase();
};

/**
 * Extracts data source information from a URI pattern
 * @param uri - URI in format "type://name/path"
 * @returns Object containing type and name
 */
exports.extractDatasetTypeFromUri = extractDatasetTypeFromUri;
const extractDataSourceInfoFromUri = uri => {
  if (!uri) return {};
  if (uri.includes('://')) {
    const parts = uri.split('://');
    if (parts.length >= 2) {
      const type = parts[0].toUpperCase();
      const pathParts = parts[1].split('/');
      const name = pathParts[0];
      return {
        type,
        name
      };
    }
  }
  return {
    name: uri
  };
};

/**
 * Constructs a data source URI from type and name
 * @param type - Dataset type (e.g., "S3", "INDEX_PATTERN")
 * @param name - Data source name
 * @returns URI in format "type://name"
 */
exports.extractDataSourceInfoFromUri = extractDataSourceInfoFromUri;
const constructDataSourceUri = (type, name) => {
  return `${type.toLowerCase()}://${name}`;
};

/**
 * Gets the dataset type from a data source reference
 * @param dataSourceRef - The data source reference object
 * @returns The dataset type or undefined
 */
exports.constructDataSourceUri = constructDataSourceUri;
const getDatasetTypeFromReference = dataSourceRef => {
  if (!(dataSourceRef !== null && dataSourceRef !== void 0 && dataSourceRef.name)) {
    return undefined;
  }
  return extractDatasetTypeFromUri(dataSourceRef.name);
};
exports.getDatasetTypeFromReference = getDatasetTypeFromReference;